//////////////////////////////////////////////////////////////////////////////
//
//  Copyright 2017 Autodesk, Inc.  All rights reserved.
//
//  Use of this software is subject to the terms of the Autodesk license 
//  agreement provided at the time of installation or download, or which 
//  otherwise accompanies this software in either electronic or hard copy form.   
//
//////////////////////////////////////////////////////////////////////////////

#pragma once

#include <plugapi.h>
#include <ifnpub.h>

#include "ai_string.h"

#include <vector>
#include <set>

#define MAXTOA_TranslationInterface_ID Interface_ID(0x27276fed, 0x52b32bac)
#define MAXTOA_DualTranslationInterface_ID Interface_ID(0x27276f66, 0x52b32b66)

struct AtNode;

namespace Max
{
    namespace ArnoldTranslator
    {
        /*! MAXtoA Arnold Translation Interface.

        This interface provides access to the MAXtoA translation code. A pointer to the implementation is passed to the   
        Translate and TranslateKeyframe methods of the third party MAXtoA_TranslationInterface implementation.

        */
        struct Arnold_translation
        {
        public:
            enum NodeType
            {
                NodeType_Texmap = 0,
                NodeType_Material = 1,
                NodeType_Camera = 2,
                NodeType_Light
            };

            enum GenericTranslationSkipFlag
            {
                SkipFlag_None = 0,
                SkipFlag_Transform = 1,
                SkipFlag_Material = 2,
                SkipFlag_ObjectProperties = 4
            };

            enum TranslationInterface_Result
            {
                //! Translation wa successful (for any reason)
                Success,
                //! Translation failed (for any reason)
                Failure,
                //! Translation has been aborted, as requested by the user or the system.
                Aborted
            };

            /*! Deletes all outputs.
            Note that a call to the method removes (and invalidates) all output in MAXtoA and deletes the nodes in the Arnold Renderer
            */
            virtual void ResetAllOutputs() = 0;

            /*! Deletes the output at the given index, such that GetOutput() will henceforth return null. 
            Note that a call to the method removes (and invalidates) the output in MAXtoA and deletes the node in the Arnold Renderer
            \param index The index of the output to delete.
            */
            virtual void ResetOutput(unsigned int index) = 0;

            /*! Initializes (creates or finds) an output node.
            The output is automatically discarded and re-created if the output already exists but refers to a node of a different type or class.
            \param index The output index of the translator (zero based).
            \param type The arnold node type 
            \param name The name of the node (to be used by Arnold)
            \ return An Arnold node (null if it could not be found/translated. 
            */
            virtual AtNode* GetOutput(unsigned int index, const AtString& type, const AtString& name) = 0;
            
            /*! Gets an Arnold node referemced by the nodes(s) currently being translated.
            If the referenced node is not already translated, this method will translate it and  return a pointer to the atNode.
            \param type A node type entry (Texmap, Material, Camera, Light).
            \param ref A pointer to the reference entity (Texmap, Material or INode* for Camera and Light).
            \ return An Arnold node (null if it could not be found/translated.
            */
            virtual AtNode* GetNode(NodeType type, ReferenceTarget* ref) = 0;
            
            /*
            
            */
            virtual Arnold_translation::TranslationInterface_Result PerformGenericTranslation(unsigned int outputIndex, int skip_flags) = 0;

            /*

            */
            virtual Arnold_translation::TranslationInterface_Result PerformGenericTranslation(unsigned int outputIndex, int skip_flags, Mtl* displacement) = 0;


            /*! Evaluates and returns the transform matrices for this INode.
            The evaluation takes into account the current motion blur parameters, from the INode and from the scene.
            The two transforms will be identical if and only motion blur is disabled for the INode.
            The evaluation will take place for the current frame time.
            \return the transformation matrices
            */
            virtual std::vector<Matrix3> EvaluateTransforms() = 0;
            
            /*! Gets the number of transformation key frames for this translation.
            \return the number of transformation key frames.
            */
            virtual unsigned int GetNbTransfKeys() = 0;

            /*! Gets the number of deformation key frames for this translation.
            \return the number of deformation key frames.
            */
            virtual unsigned int GetNbDefKeys() = 0;

            /*! Gets the TimeValue for a transformation key frame.
            \return the TimeValue for a transformation key frame.
            */
            virtual int GetTransfKey(const int in_index, const int in_t) = 0;

            /*! Gets the TimeValue for a deformation key frame.
            \return the TimeValue for a deformation key frame.
            */
            virtual int GetDefKey(const int in_index, const int in_t) = 0;
        };


        /*! MAXtoA Arnold Translation Interface.

        This interface must be present in objects that need to be picked up by the MAXtoA Generic Translator.
        Your plugin object should implement the following method:
        
        BaseInterface* YourPlugin::GetInterface(Interface_ID id)
        {
            if (id == MAXTOA_TranslationInterface_ID) {
                return path_to your_implementation_of_MAXTOA_TranslationInterface;
            }
            return YourBaseClass::GetInterface(id);
        }

        Your implementation must provide a Translate and TranslateKeyframe method.

        */
        struct MAXtoA_TranslationInterface : public BaseInterface
        {
        public:

            /*! Performs the translation for which this class is responsible.
            This method is called by the Generic Translator of the MAXtoA api and is responsible to create Arnold Node(s) and set parameters
            \param node The MAX INode being translated.
            \param translationTime The time at which the scene is being translated.
            \param [in, out] newValidity The validity of the translator, to be intersected with the input value. The translator will be re-translated
            whenever the current time value falls outside of this validity.
            \return The (success) state of the translation process. */
            virtual Arnold_translation::TranslationInterface_Result Translate( INode* node,
                Arnold_translation* arnold_translation,
                const TimeValue translationTime, 
                Interval& newValidity) = 0;

            /*! Performs the translation for a single deformation keyframe.
            This method is called by the Generic Translator of the MAXtoA api and is responsible to create Arnold Node(s) and set parameters
            \param node The MAX INode being translated.
            \param frame_time The time value at which the scene is being translated - equal to the 't' argument passed to Translate().
            \param keyframe_time The time value at which the current keyframe is to be evaluated. This is one of the values
            returned through the keyframesNeeded argument of Translate(); TranslateKeyframe() will be called once for each
            keyframe returned through keyframesNeeded, in ascending order.
            \param keyframe_index The index of the keyframe in the current translation (zero based)
            \return The (success) state of the translation process. */
            virtual Arnold_translation::TranslationInterface_Result TranslateKeyframe(INode* node,
                const TimeValue frame_time,
                const TimeValue keyframe_time,
                unsigned int keyframe_index) 
            {
                return Arnold_translation::Success;
            }

            virtual Interval CheckValidity(const TimeValue t, const Interval& previous_validity) const
            {
                Interval evaluation_interval = FOREVER;
                return (evaluation_interval & previous_validity);
            }
        };
    }
}